/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Standard Includes

	#include <stdio.h>
	#include <string.h>
	#include <stdlib.h>
	#include <assert.h>
	#include <math.h>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Buzz Includes

	#include "c:\buzz\dev\MachineInterface.h"

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Defines

	#define machine_name	"11-A2M v1.0"
	#define machine_short	"A2M "

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Machine Defines

	#define mpar_gai		"Gain",			"Gain"
	#define mpar_ofs		"Offset",		"Offset"
	#define mpar_snd		"Send",			"Send"
	#define mpar_ctr		"Constant",		"Constant"
	#define mpar_spd		"Speed",		"Speed"

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Machine Parameters

	CMachineParameter const para_gai	= {pt_byte,mpar_gai,  0,126,0xFF,MPF_STATE, 63};
	CMachineParameter const para_ofs	= {pt_byte,mpar_ofs,  0,127,0xFF,MPF_STATE,  0};
	CMachineParameter const para_snd	= {pt_byte,mpar_snd,  0,  5,0xFF,MPF_STATE,  0};
	CMachineParameter const para_ctr	= {pt_byte,mpar_ctr,  0,127,0xFF,MPF_STATE,  0};
	CMachineParameter const para_spd	= {pt_byte,mpar_spd,  1,128,0xFF,MPF_STATE, 16};

	CMachineParameter const *pParameters[]	= {&para_gai,&para_ofs,&para_snd,&para_ctr,&para_spd};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Attributes

	CMachineAttribute const attr_dev		= {"Device",0,15,4};
	CMachineAttribute const attr_chl		= {"Channel",0,15,0};

	CMachineAttribute const *pAttributes[]	= {&attr_dev,&attr_chl};

////////////////////////////////////////////////////////////////////////////////

	#pragma pack(1)		

	class gvals	{ public: byte	gai,ofs,snd,ctr,spd; };
	class avals { public: int	dev,chl; };
	
	#pragma pack()

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Machine Info

	CMachineInfo const MacInfo = 
	{
		MT_EFFECT,			// type
		MI_VERSION,	
		MIF_NO_OUTPUT,		// flags
		0,					// min tracks
		0,					// max tracks
		5,					// numGlobalParameters
		0,					// numTrackParameters
		pParameters,
		2,
		pAttributes,
		machine_name,
		machine_short,
		"Holger Zwar",
		"About"
	};

	class mi;

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Machine Interface

	class mi : public CMachineInterface
	{
		public:
			mi();
			virtual ~mi();
			virtual void Init(CMachineDataInput * const pi);
			virtual void Tick();
			virtual void Stop();
			virtual bool Work(float *psamples, int numsamples, int const mode);
			virtual char const *DescribeValue(int const param, int const value);
			virtual void Command(int const i);

		public:
			gvals	gval;
			avals	aval;
		
			float	gai;
			byte	ofs;
			byte	snd;
			byte	ctr;
			byte	spd;
			int		smp;

			byte	key;
			byte	old;
			int		cnt;

	};


	DLL_EXPORTS

	mi::mi()
	{
		GlobalVals	= &gval;
		AttrVals	= (int *)&aval;
	}

	mi::~mi()
	{
	}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	mi Init

	void mi::Init(CMachineDataInput * const pi)
	{
		old = 128;	// means not valid 
		key = 128;
	}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	mi Command

	void mi::Command(int const i)
	{
		pCB->MessageBox("coded in 2000 by z.war\n\nthx to oskari tammelin");
	}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	mi DescribeValue

	char const *mi::DescribeValue(int const param, int const value)
	{
		
		static char txt[16];

		switch(param)
		{
		case 0:
			sprintf(txt, "%.1f%%", (double)value*(200.0/126.0));
			break;
		case 2:
			switch(value)
			{
			case 0: sprintf(txt,"Controller");		break;
			case 1: sprintf(txt,"Velocity");		break;
			case 2: sprintf(txt,"Note On");			break;
			case 3: sprintf(txt,"Pitch Bend");		break;
			case 4: sprintf(txt,"Aftertouch");		break;
			case 5: sprintf(txt,"Prog. Change");	break;
			}
			break;
		case 3:
			switch(snd)
			{
			case 0: sprintf(txt,"%d (Number)",value);	break;
			case 1: sprintf(txt,"%d (Note)",value);		break;
			case 2: sprintf(txt,"%d (Velo.)",value);	break;
			default:
				return NULL;
			}
			break;
		case 4:
			sprintf(txt, "%d:%d Ticks",(int)value/16,(int)value%16);
			break;
		default:
			return NULL;
		}

		return txt;

	}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	mi Tick

	void mi::Tick()
	{

		if (gval.gai != para_gai.NoValue)			// input gain
			gai = (float)(gval.gai*1.0/63);

		if (gval.ofs != para_ofs.NoValue)			// offset
			ofs = gval.ofs;
		
		if (gval.spd != para_spd.NoValue)			// speed
			spd = gval.spd;

		smp = spd*pMasterInfo->SamplesPerTick/16;	// speed in samples, calculated every tick (if master speed changes)

		if (gval.snd != para_snd.NoValue)			// what message to send
		{
			snd = gval.snd;
			cnt = smp;
		}
		
		if (gval.ctr != para_ctr.NoValue)			// variable value (controller, key, velocity)
			ctr = gval.ctr;
		
	}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	mi Stop

	void mi::Stop()
	{

		if (key!=128)	// note off if playing
		{
			dword out = key * 0x100 + (0x90 + aval.chl);
			key = 128;
			pCB->MidiOut(aval.dev, out);		
		}

	}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	mi Work

bool mi::Work(float *psamples, int numsamples, int const mode)
{

	if (mode == WM_WRITE || mode == WM_NOIO)
		return false;

	if (mode == WM_READ )
		return true;		

	dword out;

	if (pCB->GetStateFlags() != NULL)	// only if playing or recording
	{

		do 

		{

			if (cnt++>=smp)	// a new tick
			{

				cnt=0;

				int val = int(*psamples * gai);	// we take the positive amount of input / 256
				if (val<0)
					val=-val;
				val=ofs + (val>>8);

				if (val>127)					// and clip it up to 127
					val=127;


				if (key!=128)	// note off if playing
				{
					out = key * 0x100 + (0x90 + aval.chl);
					key = 128;
					pCB->MidiOut(aval.dev, out);		
				}

				if (val!=old || (snd>=1 && snd<=2) )
				{

					out = 0;	// dirty

					switch(snd)
					{
					case 0:	// CC
						out = val * 0x10000 + ctr * 0x100 + (0xB0 + aval.chl);
						break;
					case 1:	// Vel
						if (val>0)	// we don't want to send more note off's
						{
							out = val * 0x10000 + ctr * 0x100 + (0x90 + aval.chl);
							key = ctr;
						}
						break;
					case 2:	// Key
						if (ctr>0)	// we don't want to send more note off's
						{
							out = ctr * 0x10000 + val * 0x100 + (0x90 + aval.chl);
							key = val;
						}
						break;
					case 3:	// PB (may be the formula is wrong here?)
						if (val<=0x40)
							out = val * 0x10000 + (0xE0 + aval.chl);
						else
							out = val * 0x10000 + (val-0x40)*2 * 0x100 + (0xE0 + aval.chl);
						break;
					case 4:	// AT
						out = val * 0x100 + (0xD0 + aval.chl);
						break;
					case 5:	// PC
						out = val * 0x100 + (0xC0 + aval.chl);
						break;
					}

					if (out!=0)	// only if not dirty
						pCB->MidiOut(aval.dev, out);

					old = val;

				}

			}

			psamples++;

		}
		while(--numsamples);

	}

	return true;

}

