/******************************************************************************
PixelFreqMap

  Given a pixel position (in the spectrogram or frequency control) return
  frequency fraction (0..1)

  Uses buzz param "freq_value" function to calculate frequency values

  Result is linear to exponential mapping


History
	Date       Version    Programmer         Comments
	16/2/03    1.0        Darrell Tam		Created
******************************************************************************/

#ifndef _PIXEL_TO_FREQ_FRAC_H_
#define _PIXEL_TO_FREQ_FRAC_H_
#include <vector>
#include "params.h"
#include "misc_stuff.h"

//------------------------------------------------------------------------------------------
class PixelFreqMap
{
public:
	// range of bin positions for each pixel
	struct BinRng { int start, end; };

	PixelFreqMap();
	virtual ~PixelFreqMap();

	// generate mapings
	void build(int n_pixels_, int sample_freq_);
	
	int getNPixels(void) const { return n_pixels; }

	// access methods
	const float&/*0..1*/PixelFreqMapFrac(int pixel) const
		{ return pixel_to_freqfrac[limit_range(pixel, 0, n_pixels-1)]; }

	float/*hz*/pixelToFreq(float pixel) const
		{ return PixelFreqMapFrac(pixel)*sample_freq/2; }

	float/*hz*/pixelFracToFreq(float/*0..1*/frac) const
		{ return pixelToFreq(frac*(n_pixels-1)); }

	// return pixel to bin for a particular block length, array length is n_pixels+1
	const BinRng* pixelToBinRng(int/*log2(fft blk len/BLK_SZ_0)*/i) const
		{ return pixel_to_binrng[i].begin(); }


	// maximum buzz value for pixel map
	float maxBuzzVal(void) const
		{ return max_buzz_val.get(); }

	// number of pixels per octave
	float pixelsPerOctave() { return (n_pixels-1)*_FreqParam::BUZZ_OCTAVE/maxBuzzVal(); }

protected:
	// sample rate in Hz
	int sample_freq;

	// maximum frequency of sample_rate/2 value as per Buzz (note*256)
	_FreqParam max_buzz_val;

	// pixels wide
	int n_pixels;

	// pixel to freq-frac (0.0=0 Hz, 1.0=sample_freq/2)
	vector<float> pixel_to_freqfrac;

	// map pixel to bin for each fft length
	// frng[blk len][pixel]
	struct VecBinRng : public vector<BinRng> {}; // do this to stop MSVC debug complaining
	vector<VecBinRng> pixel_to_binrng;
};

#endif
