/******************************************************************************
DtEdit

Windowless text edit control that can be drawn on top of graphics

History
	Date       Version    Programmer         Comments
	29/07/03   1.0        Darrell Tam
******************************************************************************/

#if !defined(AFX_DTEDIT_H__CFCD3BB9_1BDB_455E_B765_35FACF77A390__INCLUDED_)
#define AFX_DTEDIT_H__CFCD3BB9_1BDB_455E_B765_35FACF77A390__INCLUDED_

#include "StdAfx.h"
#include "misc_stuff.h"

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000
// DtEdit.h : header file
//

/////////////////////////////////////////////////////////////////////////////
// DtEdit window



//----------------------------------------------------------------------------------------------------
class DtEdit
{
// Construction
public:

	DtEdit();
	virtual ~DtEdit();

	// changes are not automatically redrawn
	void setFont(CFont *font_);
	void setInsideColour(COLORREF c) { setValue(inside_colour, c, require_redraw); }
	void setOutsideColour(COLORREF c) { setValue(outside_colour, c, require_redraw); }
	void setColour(COLORREF inside, COLORREF outside)
		{ setInsideColour(inside); setOutsideColour(outside); }

	// set text, no auto redraw
	void setText(const string& txt_);

	// note, if text is changed call requireRedraw() to invalidate
	string& getText_(void) { return txt; } 
	const string& getText(void) const { return txt; }
	operator const char* () const { return txt.c_str(); }

	// drawing functions
	void requireRedraw(bool x) { require_redraw = x; }
	bool requireRedraw() const { return require_redraw; }

	CSize size(CDC *dc);
	void drawCursor(CDC* dc, CPoint pos/*top-left corner of text*/);
	void drawText(CDC* dc, CPoint pos/*top-left corne of text*/);
	void draw(CDC* dc, CPoint pos)
	{
		drawCursor(dc, pos);
		drawText(dc, pos);
	}

	// changes are not auto redrawn
	void cursorVisible(bool t) { setValue(cursor_visible, t, require_redraw); }
	void setCursorPos(int cursor_pos_, bool select_mode/*shift key*/);
	void setCursorPos(int cursor_pos_);
	void setSelectPos(int select_pos_);

	// query cursor position
	int getCursorSelectMin(void) const { return min(cursor_pos, select_pos); }
	int getCursorSelectMax(void) const { return max(cursor_pos, select_pos); }
	int getCursorPos(void) const { return cursor_pos; }
	int getSelectPos(void) const { return select_pos; }
	int isSelection(void) const { return select_pos != cursor_pos; }
	bool isCursorHome(void) const { return !isSelection() && cursor_pos==0; }
	bool isCursorEnd(void) const { return !isSelection() && cursor_pos==txt.size(); }
	bool isAllSelected(void) const { return getCursorSelectMin()==0 && getCursorSelectMax()==txt.size();}

	// changes are not auto redrawn
	void selectAll(bool cursor_at_end);

	// text edit keys
	void insert(string s);
	void backSpaceKey(void);
	void deleteKey(void);

	// cursor movement keys
	void endKey(bool shift_state=false) { setCursorPos(txt.size(), shift_state); }
	void homeKey(bool shift_state=false) { setCursorPos(0, shift_state); }
	void leftKey(bool shift_state=false) { setCursorPos(cursor_pos-1, shift_state); }
	void rightKey(bool shift_state=false) { setCursorPos(cursor_pos+1, shift_state); }

	// process keyboard events
	void doKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags);
	void doKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags);

protected:
	// text to display
	string txt;	

	int select_pos; // highlight position
	int cursor_pos;	// note, characters are inserted before cursor
	bool cursor_visible;	// whether cursor is visible
	bool require_redraw;	// needs redraw

	CFont* font;
	COLORREF inside_colour;
	COLORREF outside_colour;

	// pixel width of text
	int txtWidth(CDC* dc, char* s, int n);

	// limitCursor bounds cursor/selection to text
	void limitCursor(void);

	void fillFontDetails(CDC *dc, bool force=false);

	int char_width[256];
	bool char_width_valid;

	TEXTMETRIC text_metric;

};


#endif // !defined(AFX_DTEDIT_H__CFCD3BB9_1BDB_455E_B765_35FACF77A390__INCLUDED_)
